using System;
using System.Collections.Generic;
using System.Text;

namespace GMDL
{
    // http://atlas.nrcan.gc.ca/site/english/Proj.js

    class ToporamaProjJS
    {
        /*
        Author:       Mike Adair mike.adairATccrs.nrcan.gc.ca
        License:      LGPL as per: http://www.gnu.org/copyleft/lesser.html
        $Id: Proj.js 735 2005-08-25 21:17:26Z jdoyon $
        */

        /**
         * Provides latitude/longitude to map projection (and vice versa) transformation methods. 
         * Initialized with EPSG codes.  Has properties for units and title strings.
         * All coordinates are handled as points which is a 2 element array where x is 
         * the first element and y is the second. 
         * For the Forward() method pass in lat/lon and it returns map XY.
         * For the Inverse() method pass in map XY and it returns lat/long.
         *
         * TBD: retrieve initialization params (and conversion code?) from a web service 
         *
         * @constructor
         *
         * @param srs         The SRS (EPSG code) of the projection
         */

        public ToporamaProjJS(string srs)
        {
            lccinit(new double[] { 6378137.0, 6356752.314, 49.0, 77.0, -95.0, 49.0, 0.0, 0 });
        }


/****************************************************************************
The following code is a direct port of PROJ4 coordinate transformation code
from C to Javascript.  For more information go to http://proj.maptools.org/
Currently suppported projections include: Lambert Conformal Conic (LCC), 
Lat/Long, Polar Stereographic.
Porting C to Javascript is fairly straightforward so other support for more 
projections is easy to add.
*/

        double PI = Math.PI;
        double HALF_PI = Math.PI * 0.5;
        double TWO_PI = Math.PI * 2.0;
        double EPSLN = 1.0e-10;
        double R2D = 57.2957795131;
        double D2R = 0.0174532925199;
        //double R = 6370997.0;        // Radius of the earth (sphere)


      double r_major;
      double r_minor;

      double center_lon;
      double center_lat;
      double false_easting;
      double false_northing;

        double e;
        double ns;
        double f0;
        double rh;
        double rh1;

        // Lambert Conformal Conic inverse equations--mapping x,y to lat/long
// -----------------------------------------------------------------
        public double[] lcc2ll(double x, double y)
        {

            double rh1, con, ts;
            double lat, lon;
            x = x - this.false_easting;
            y = this.rh - y + this.false_northing;
            if (this.ns > 0)
            {
                rh1 = Math.Sqrt(x * x + y * y);
                con = 1.0;
            }
            else
            {
                rh1 = -Math.Sqrt(x * x + y * y);
                con = -1.0;
            }
            double theta = 0.0;
            if (rh1 != 0)
            {
                theta = Math.Atan2((con * x), (con * y));
            }
            if ((rh1 != 0) || (this.ns > 0.0))
            {
                con = 1.0 / this.ns;
                ts = Math.Pow((rh1 / (this.r_major * this.f0)), con);
                lat = phi2z(this.e, ts);
                if (lat == -9999) return null;
            }
            else
            {
                lat = -HALF_PI;
            }
            lon = adjust_lon(theta / this.ns + this.center_lon);
            return new double[] { R2D * lon, R2D * lat };
        }

        // Function to compute the latitude angle, phi2, for the inverse of the
//   Lambert Conformal Conic and Polar Stereographic projections.
// ----------------------------------------------------------------
        private double phi2z(double eccent, double ts)
        {
            double eccnth = .5 * eccent;
            double con, dphi;
            double phi = HALF_PI - 2 * Math.Atan(ts);
            for (int i = 0; i <= 15; i++)
            {
                con = eccent * Math.Sin(phi);
                dphi = HALF_PI - 2 * Math.Atan(ts * (Math.Pow(((1.0 - con) / (1.0 + con)), eccnth))) - phi;
                phi += dphi;
                if (Math.Abs(dphi) <= .0000000001) return phi;
            }
            //alert("Convergence error - phi2z");
            return -9999;
        }

// NEED!!!!
// Initialize the Lambert Conformal conic projection
// -----------------------------------------------------------------
private void lccinit(double[] param)
{
    // array of:  r_maj,r_min,lat1,lat2,c_lon,c_lat,false_east,false_north
    //double c_lat;                   /* center latitude                      */ // WGS84 fixed
    //double c_lon;                   /* center longitude                     */ // WGS84 fixed
    //double lat1;                    /* first standard parallel              */ // ?
    //double lat2;                    /* second standard parallel             */ // ?
    //double r_maj;                   /* major axis                           */ // center_lon
    //double r_min;                   /* minor axis                           */ // center_lat
    //double false_east;              /* x offset in meters                   */ // false_easting
    //double false_north;             /* y offset in meters                   */ // false_northing

  this.r_major = param[0];
  this.r_minor = param[1];
  double lat1 = param[2] * D2R;
  double lat2 = param[3] * D2R;
  this.center_lon = param[4] * D2R;
  this.center_lat = param[5] * D2R;
  this.false_easting = param[6];
  this.false_northing = param[7];

// Standard Parallels cannot be equal and on opposite sides of the equator
 // if (Math.abs(lat1+lat2) < EPSLN) {
    //alert("Equal Latitiudes for St. Parallels on opposite sides of equator - lccinit");
    //return;
 // }

  double temp = this.r_minor / this.r_major;
  this.e = Math.Sqrt(1.0 - temp*temp);

  double sin1 = Math.Sin(lat1);
  double cos1 = Math.Cos(lat1);
  double ms1 = msfnz(this.e, sin1, cos1);
  double ts1 = tsfnz(this.e, lat1, sin1);
  
  double sin2 = Math.Sin(lat2);
  double cos2 = Math.Cos(lat2);
  double ms2 = msfnz(this.e, sin2, cos2);
  double ts2 = tsfnz(this.e, lat2, sin2);
  
  double ts0 = tsfnz(this.e, this.center_lat, Math.Sin(this.center_lat));

  if (Math.Abs(lat1 - lat2) > EPSLN) {
    this.ns = Math.Log(ms1/ms2)/Math.Log(ts1/ts2);
  } else {
    this.ns = sin1;
  }
  this.f0 = ms1 / (this.ns * Math.Pow(ts1, this.ns));
  this.rh = this.r_major * this.f0 * Math.Pow(ts0, this.ns);
}

        // NEED!!!
        // Lambert Conformal conic forward equations--mapping lat,long to x,y
        // -----------------------------------------------------------------
        public double[] ll2lcc(double lon, double lat)
        {
            // convert to radians
            if ( lat <= 90.0 && lat >= -90.0 && lon <= 180.0 && lon >= -180.0)
            {
                lat *= D2R;
                lon *= D2R;
            }
            else
            {
                //alert("*** Input out of range ***: lon: "+lon+" - lat: "+lat);
                return null;
            }

            double con  = Math.Abs( Math.Abs(lat) - HALF_PI);
            double ts;
            if (con > EPSLN)
            {
                ts = tsfnz(this.e, lat, Math.Sin(lat));
                rh1 = this.r_major * this.f0 * Math.Pow(ts, this.ns);
            }
            else
            {
                con = lat * this.ns;
                if (con <= 0)
                {
                    //alert("Point can not be projected - ll2lcc");
                    return null;
                }
                rh1 = 0;
            }
            double theta = this.ns * adjust_lon(lon - this.center_lon);
            double x = rh1 * Math.Sin(theta) + this.false_easting;
            double y = this.rh - rh1 * Math.Cos(theta) + this.false_northing;

            return new double[] { x, y };
        }

// Function to compute the constant small m which is the radius of
//   a parallel of latitude, phi, divided by the semimajor axis.
// -----------------------------------------------------------------
private double msfnz(double eccent, double sinphi, double cosphi) {
      double con = eccent * sinphi;
      return cosphi/(Math.Sqrt(1.0 - con * con));
}

// Function to compute the constant small t for use in the forward
//   computations in the Lambert Conformal Conic and the Polar
//   Stereographic projections.
// -----------------------------------------------------------------
private double tsfnz(double eccent, double phi, double sinphi)
{
  double con = eccent * sinphi;
  double com = .5 * eccent; 
  con = Math.Pow(((1.0 - con) / (1.0 + con)), com);
  return (Math.Tan(.5 * (HALF_PI - phi))/con);
}


// Function to return the sign of an argument
private double sign(double x) { if (x < 0.0) return(-1); else return(1);}

// Function to adjust longitude to -180 to 180; input in radians
private double adjust_lon(double x) {x=(Math.Abs(x)<PI)?x:(x-(sign(x)*TWO_PI));return(x);}

    }
}
